@extends('member-panel.layouts.app')

@section('title', 'Balance Transfer')

@push('css')
<link rel="stylesheet" href="{{ asset('public/assets/member-panel/assets/css/toastr.css') }}?{{ time() }}">
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.0/font/bootstrap-icons.css">

<style>
    :root {
        --primary: #4f46e5;
        --primary-hover: #4338ca;
        --success: #10b981;
        --danger: #ef4444;
        --text: #1f2937;
        --text-muted: #6b7280;
        --background: #f9fafb;
        --card-bg: #ffffff;
    }



    .transfer-container {
        padding: 2rem 1rem;
        min-height: 100vh;
        display: flex;
        align-items: flex-start;
        justify-content: center;
    }

    .transfer-card {
        max-width: 600px;
        width: 100%;

        border-radius: 16px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
        overflow: hidden;
        animation: fadeInUp 0.5s ease-out;
    }


    .card-header {


        padding: 1.5rem;
        text-align: center;
    }
    .card-header h2 {
        font-size: 1.5rem;
        font-weight: 700;
        margin-bottom: 0.5rem;
    }

    .balance-display {

        border-radius: 10px;
        padding: 1rem;
        display: inline-flex;
        align-items: center;
        gap: 0.75rem;
    }

    .balance-amount {
        font-size: 1.5rem;
        font-weight: 600;
    }

    .card-body {
        padding: 2rem;
    }

    .form-group {
        margin-bottom: 1.5rem;
    }

    .form-label {
        font-weight: 600;
        font-size: 0.875rem;
        margin-bottom: 0.5rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .form-control {
        border: 1px solid #d1d5db;
        border-radius: 8px;
        padding: 0.75rem 1rem;
        font-size: 1rem;
        width: 100%;
        transition: all 0.2s ease;
    }

    .form-control:focus {
        outline: none;
        border-color: var(--primary);
        box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
    }

    .input-icon {
        position: relative;
    }

    .input-icon i {
        position: absolute;
        left: 1rem;
        top: 50%;
        transform: translateY(-50%);
        color: var(--text-muted);
        font-size: 1.1rem;
    }

    .input-icon .form-control {
        padding-left: 2.75rem;
    }

    .password-container {
        position: relative;
    }

    .password-toggle {
        position: absolute;
        right: 1rem;
        top: 50%;
        transform: translateY(-50%);
        cursor: pointer;
        color: var(--text-muted);
        transition: color 0.2s ease;
    }

    .password-toggle:hover {
        color: var(--primary);
    }

    .amount-breakdown {

        border-radius: 10px;
        padding: 1rem;
        margin-bottom: 1.5rem;
    }

    .breakdown-title {
        font-size: 1rem;
        font-weight: 600;
        margin-bottom: 0.75rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .breakdown-item {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 0.5rem 0;
        font-size: 0.95rem;
    }

    .breakdown-item:last-child {
        border-top: 1px solid #d1d5db;
        padding-top: 0.75rem;
        margin-top: 0.75rem;
        font-weight: 600;
    }

    .breakdown-value {
        font-weight: 600;
    }

    .breakdown-item:last-child .breakdown-value {
        color: var(--success);
    }

    .submit-btn {

        color: #ffffff;
        border: none;
        border-radius: 8px;
        padding: 1rem;
        font-size: 1rem;
        font-weight: 600;
        width: 100%;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 0.5rem;
    }

    .submit-btn:hover:not(:disabled) {

        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(79, 70, 229, 0.3);
    }

    .submit-btn:disabled {

        cursor: not-allowed;
    }

    .submit-btn.loading::after {
        content: '';
        width: 20px;
        height: 20px;
        border: 2px solid #ffffff;
        border-top-color: transparent;
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }

    @keyframes fadeInUp {
        from { opacity: 0; transform: translateY(20px); }
        to { opacity: 1; transform: translateY(0); }
    }

    @keyframes spin {
        to { transform: rotate(360deg); }
    }

    @media (max-width: 576px) {
        .transfer-container {
            padding: 1rem;
        }

        .card-body, .card-header {
            padding: 1.5rem;
        }

        .balance-amount {
            font-size: 1.25rem;
        }
    }
</style>
@endpush

@section('content')
<div class="transfer-container">
    <div class="transfer-card card adminuiux-card shadow-sm mb-4">
        <div class="card-header">
            <h2>
                <i class="bi bi-arrow-left-right me-2"></i>Balance Transfer
            </h2>
            <p class="mb-0">Transfer funds between your wallets or to another user</p>
            <div class="balance-display">
                <i class="bi bi-piggy-bank fs-4"></i>
                <div>
                    <p class="mb-1 text-white opacity-75">Available Balance</p>
                    <div class="balance-amount">
                        {{ number_format(auth('customer')->user()->income_wallet, 2) }}
                        <span class="fs-6">{{ $web_config['currency_sign']->value }}</span>
                    </div>
                </div>
            </div>
        </div>

        <div class="card-body">
            <form action="{{ url('member/balance-transfer') }}" method="post" id="transferForm">
                @csrf

                <div class="form-group">
                    <label class="form-label" for="wallet_type">
                        <i class="bi bi-wallet2"></i>Transfer To
                    </label>
                    <div class="input-icon">
                        <i class="bi bi-list-check"></i>
                        <select class="form-control" name="wallet_type" id="wallet_type" required aria-describedby="walletHelp">
                            <option value="" disabled>Select Transfer Wallet Type</option>
                            <option value="1">Income Wallet</option>
                            <option value="2" selected>Purchase Wallet</option>
                        </select>
                    </div>
                    <small id="walletHelp" class="text-muted">Choose the destination wallet</small>
                    @error('wallet_type')
                        <div class="text-danger text-sm mt-1">{{ $message }}</div>
                    @enderror
                </div>

                <div class="form-group user-id-group" style="display: none;">
                    <label class="form-label" for="user_id">
                        <i class="bi bi-person"></i>Recipient Username
                    </label>
                    <div class="input-icon">
                        <i class="bi bi-person-circle"></i>
                        <input type="text" name="user_id" id="user_id" class="form-control" placeholder="Enter recipient username">
                    </div>
                    @error('user_id')
                        <div class="text-danger text-sm mt-1">{{ $message }}</div>
                    @enderror
                </div>

                <div class="form-group">
                    <label class="form-label" for="final_amount">
                        <i class="bi bi-currency-dollar"></i>Amount
                    </label>
                    <div class="input-icon">
                        <i class="bi bi-cash"></i>
                        <input type="number" name="final_amount" id="final_amount" class="form-control" placeholder="Enter amount" min="0" step="0.01" required aria-describedby="amountHelp">
                    </div>
                    <small id="amountHelp" class="text-muted">Enter the amount to transfer</small>
                    @error('final_amount')
                        <div class="text-danger text-sm mt-1">{{ $message }}</div>
                    @enderror
                </div>

                <div class="form-group">
                    <label class="form-label" for="password">
                        <i class="bi bi-shield-lock"></i>Transaction Password
                    </label>
                    <div class="input-icon password-container">
                        <i class="bi bi-lock"></i>
                        <input type="password" name="password" id="password" class="form-control" placeholder="Enter transaction password" required>
                        <span class="password-toggle" id="togglePassword" aria-label="Toggle password visibility">
                            <i class="bi bi-eye"></i>
                        </span>
                    </div>
                    @error('password')
                        <div class="text-danger text-sm mt-1">{{ $message }}</div>
                    @enderror
                </div>

                <div class="amount-breakdown">
                    <div class="breakdown-title">
                        <i class="bi bi-calculator"></i>Summary
                    </div>
                    <div class="breakdown-item">
                        <span>Transfer Amount</span>
                        <span class="breakdown-value" id="entered_amount">0.00 {{ $web_config['currency_sign']->value }}</span>
                    </div>
                    <div class="breakdown-item">
                        <span>Transaction Fee (0%)</span>
                        <span class="breakdown-value">0.00 {{ $web_config['currency_sign']->value }}</span>
                    </div>
                    <div class="breakdown-item">
                        <span>Total Deduction</span>
                        <span class="breakdown-value" id="deducted_amount">0.00 {{ $web_config['currency_sign']->value }}</span>
                    </div>
                </div>

                <button type="submit" class="submit-btn" id="submitBtn" disabled>
                    <i class="bi bi-send"></i>Send Transfer
                </button>
            </form>
        </div>
    </div>
</div>
@endsection

@push('js')
<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
<script src="{{ asset('public/assets/member-panel/assets/js/toastr.js') }}?{{ time() }}"></script>
{!! Toastr::message() !!}

<script>
$(document).ready(function() {
    toastr.options = {
        positionClass: 'toast-top-right',
        timeOut: 5000,
        closeButton: true,
        progressBar: true,
        preventDuplicates: true
    };

    $('#togglePassword').on('click', function() {
        const $passwordField = $('#password');
        const $icon = $(this).find('i');
        const type = $passwordField.attr('type') === 'password' ? 'text' : 'password';
        $passwordField.attr('type', type);
        $icon.toggleClass('bi-eye bi-eye-slash');
    });

    $('#wallet_type').on('change', function() {
        const walletType = $(this).val();
        const $userIdGroup = $('.user-id-group');
        const $userId = $('#user_id');

        if (walletType === '1') {
            $userIdGroup.slideUp(200);
            $userId.prop('required', false).val('');
        } else if (walletType === '2') {
            $userIdGroup.slideDown(200);
            $userId.prop('required', true);
        }
        validateForm();
    }).trigger('change');

    $('#final_amount, #user_id, #password').on('input change', validateForm);

    $('#final_amount').on('input', function() {
        const amount = parseFloat($(this).val()) || 0;
        const balance = parseFloat('{{ auth('customer')->user()->income_wallet }}');
        if (amount > balance) {
            toastr.warning('Amount exceeds available balance');
            $(this).val(balance);
        }
        if (amount < 0) {
            $(this).val(0);
        }
        calculateAmount();
        validateForm();
    });

    $('#transferForm').on('submit', function(e) {
        const amount = parseFloat($('#final_amount').val()) || 0;
        const balance = parseFloat('{{ auth('customer')->user()->income_wallet }}');
        const walletType = $('#wallet_type').val();

        if (amount <= 0) {
            e.preventDefault();
            toastr.error('Please enter a valid amount');
            return false;
        }
        if (amount > balance) {
            e.preventDefault();
            toastr.error('Insufficient balance for transfer');
            return false;
        }
        if (walletType === '2' && !$('#user_id').val().trim()) {
            e.preventDefault();
            toastr.error('Please enter a recipient username');
            return false;
        }

        const $submitBtn = $('#submitBtn');
        $submitBtn.addClass('loading').prop('disabled', true);
        $submitBtn.html('<i class="bi bi-hourglass-split me-1"></i> Processing...');

        return confirm(`Confirm transfer of ${amount.toFixed(2)} {{ $web_config['currency_sign']->value }}?`);
    });
});

function calculateAmount() {
    const amount = parseFloat($('#final_amount').val()) || 0;
    $('#entered_amount').text(`${amount.toFixed(2)} {{ $web_config['currency_sign']->value }}`);
    $('#deducted_amount').text(`${amount.toFixed(2)} {{ $web_config['currency_sign']->value }}`);
}

function validateForm() {
    const walletType = $('#wallet_type').val();
    const amount = parseFloat($('#final_amount').val()) || 0;
    const balance = parseFloat('{{ auth('customer')->user()->income_wallet }}');
    const userId = $('#user_id').val().trim();
    const password = $('#password').val().trim();

    const isValid = walletType &&
                    amount > 0 &&
                    amount <= balance &&
                    password.length > 0 &&
                    (walletType === '1' || (walletType === '2' && userId.length > 0));

    const $submitBtn = $('#submitBtn');
    $submitBtn.prop('disabled', !isValid);
    if (isValid) {
        $submitBtn.html('<i class="bi bi-send"></i> Send Transfer');
    }
}

validateForm();
</script>
@endpush
